class Wlsqrs extends Wlstrend {

// -------------------------------------------------------------------

static void wlsqrs_(double y[], double c[][], double s[][], int m,
   int n, double x[]) {
 
/*  Finds the weighted least squares estimate of x.  c is the design
    matrix, s is the lower triangular factor of the covariance matrix,
    i.e., w=s*s**T where w is the covariance matrix.
 
    wlsqrs calls rotn1, rotn2, rotn3
 
   The routine is translated from an Algol W code from:
   Kourouklis, S. (1977), ``Computing Weighted Linear Least
   Squares Solutions,'' Technical Report SOCS-79.16, Master of
   Science Thesis, McGill University, John MacNeal Physical
   Science and Engineering Library, 809 Sherbrooke St. W.
   Montreal, Quebec, Canada H3A2K6
 
   On return, r2 and L are the factors of the covariance matrix of xhat.
 
   Parameters:
 
   c:   input m*n matrix.  During the algorithm, c is transformed
        to l first and then possibly to lcurly
   s:   input and partially output m*m matrix.  On input s is the
        lower trangular factor of w, and on output a part of s holds r2.
   y:   input m*1 vector of observations
   m:   input scalar, row dimension of c
   n:   input scalar, column dimension of c
   x:   output n*1 vector, the estmate of xhat
   k:   output scalar, the rank of c
   t:   output scalar, the rank of r1**(t)
   ier: output scalar to indicate whether the model is wrong */

int i, j;

double q1ty, res, sum, add;

double pp[][] = new double[NMWRPAR][NMWRPAR];
 
ier = 0;
 
// Reduce (c,s).
 
rotn1_(c, s, y, m, n);

/*  solve r1**(t)*z=q1**(t)*y for z */

if (k != m) {
   if (t == (m - k)) {
      lowerb_(s,t,0,y);

   } else {
      q1ty=0.;
      for (i=0; i < m - k; ++i) q1ty += y[i] * y[i];
      q1ty= Math.sqrt(q1ty);
      rotn2_(s,y,m - k,t);
      res=0;
      for (i=0; i < m - k - t; ++i) {
	 res += y[i] * y[i];
      }
      res= Math.sqrt(res);
      add=q1ty+res;
      if (add != q1ty) {
         ier=1;
         return;
      }
      lowerb_(s,t,m - k - t,y);
   }
 
   /* Compute b=q2**(t)*y-r12**(t)*z and store it in the last k entries
      of y. */
 
   for (i=0; i < k; ++i) {
      sum=0.;
      for (j=0; j < t; ++j) {
	 sum += s[m - k+i][j] * y[m - k - t+j];
      }
      y[m - k+i] -= sum;
   }
}
 
/*  solve l*xhat=b for xhat */
 
if (k == n) {
   lowers_(c,k,m - k,y);
   for (i=0; i < n; ++i) {
      x[i]=y[m - k+i];
   }

} else {
   rotn3_(c,m,n,k,pp);
   lowers_(c,k,m - k,y);
   for (i=0; i < n; ++i) {
      x[i]=0.;
      for (j=0; j < k; ++j) x[i] += pp[i][j] * y[m - k+j];
   }
}
}

// -----------------------------------------------------------------

static void rotn1_(double c[][], double s[][], double y[], int m,
   int n) {
 
/* reduce (c,s) to (0 r1**(t)  0         )
                   (                     )
                   (l r12**(t) r2**(t)   )
  also form q**(t)*y                               */
 
int i, j, jj, kk, mm, idum;
double dumu[] = new double[WLSSZE];
double dumv[] = new double[WLSSZE];
double temp, big, max, add;

max=0.;
for (i=0; i < m; ++i) {
   for (j=0; j <= i; ++j) {
      if (Math.abs(s[i][j]) > max) max=Math.abs(s[i][j]);
   }
}

big=0.;
for (i=0; i < m; ++i) {
   for (j=0; j < n; ++j) {
      if (Math.abs(c[i][j]) > big) big=Math.abs(c[i][j]);
   }
}
 
/*  j denotes a column of c and mm the number of elements to be
    eliminated in this column */
 
j=n;
mm=m-1;
 
/*  eliminate c(i,j) carrying its weight to c(i+1,j). update s,y */
 
while ((j > 0) && (mm > 0)) {
   for (i=1; i <= mm; ++i) {
      add=c[i][j-1]+c[i-1][j-1];
      if (add != c[i][j-1]) { 
         cossin_(c[i - 1], c[i],j,1,j-1);
         s[i-1][i] = -wsin*s[i][i];
         s[i][i] = wcos*s[i][i];
         for (jj=1; jj <= i; ++jj) {
            temp=wsin*s[i-1][jj-1]+wcos*s[i][jj-1];
            s[i-1][jj-1]=wcos*s[i-1][jj-1]-wsin*s[i][jj-1];
            s[i][jj-1]=temp;
         }
         temp=wsin*y[i-1]+wcos*y[i];
         y[i-1]=wcos*y[i-1]-wsin*y[i];
         y[i]=temp;
      }
   }

// keep s lower triangular all the time
 
   for (i=1; i <= mm; ++i) {
      add=s[i-1][i-1]+s[i-1][i];
      if (add != s[i-1][i-1]) { 
         for (idum=0; idum < m; ++idum) {
            dumu[idum]=s[idum][i];
            dumv[idum]=s[idum][i-1];
         }
         cossin_(dumu,dumv,i,i+1,m);
         for (idum=0; idum < m; ++idum) {
            s[idum][i]=dumu[idum];
            s[idum][i-1]=dumv[idum];
         }
      }
   }
 
/*  check if number of eliminated elements in the current column is mm+1 */
 
   add=big+Math.abs(c[mm][j-1]);
   if (add != big) --mm;
   --j;
}

/*  kk is the number of zero rows in c */
 
kk=0;
if (m > n) kk=m-n;

l50: while (kk <= m) {
   for (j=0; j < n-m+kk+1; ++j) {
      add=big+ Math.abs(c[kk][j]);
      if (add != big) break l50;
   }
   ++kk;
}

k = m-kk;
 
/*  determine the rank of r1**(t) */
 
if (m != k) {
   t = m - k;

   while (t > 0) {
      add=max+Math.abs(s[t-1][t-1]);
      if (max != add) break;
      --t;
   }
}
}

// ---------------------------------------------------------------------

static void rotn2_(double a[][], double y[], int m, int n) {
 
/*  reduces an m*n lower trapezoidal full column rank matrix
    a to an n*n lower triangular nonwsingular matrix r */
 
int i, j;

double temp, add;

for (j=n; j >= 1; --j) {
   for (i=j; i <= m-n+j-1; ++i) {
      add=a[(i+1)-1][j-1]+a[i-1][j-1];
      if (add != a[(i+1)-1][j-1]) {
         cossin_(a[i - 1], a[(i + 1) - 1],j,1,j-1);
         temp=wsin*y[i-1]+wcos*y[(i+1)-1];
         y[i-1]=wcos*y[i-1]-wsin*y[(i+1)-1];
         y[(i+1)-1]=temp;
      }
   }
}
}

// ---------------------------------------------------------------------

static void rotn3_(double a[][], int m, int n, int k, double pp[][]) {
 
/* Reduces a k*n full row rank matrix a to a k*k lower triangular matrix
   lcurly, i.e., a*pp=(lcurly,0) where pp is an n*n orthogonal matrix.
   a has some lower trapezoidal form. */

int ii, idum, i, j;

double temp, add;

double dumu[] = new double[WLSSZE];
double dumv[] = new double[WLSSZE];

for (i=0; i < n; ++i) {
   for (j=0; j < n; ++j) {
      pp[i][j]=0.;
      if (i == j) pp[i][j]=1.;
   }
}
 
for (i=1; i <= k; ++i) {
   for (j=n-k+i-1; j >= i; --j) {
      add=a[(m-k+i)-1][j-1]=a[(m-k+i)-1][(j+1)-1];
      if (add != a[(m-k+i)-1][j-1]) {
         for (idum=1; idum <= m; ++idum) {
            dumu[idum-1]=a[idum-1][(j+1)-1];
            dumv[idum-1]=a[idum-1][j-1];
         }
         cossin_(dumu,dumv,m-k+i,m-k+i+1,m);
         for (idum=1; idum <= m; ++idum) {
            a[idum-1][(j+1)-1]=dumu[idum-1];
            a[idum-1][j-1]=dumv[idum-1];
         }
         for (ii=1; ii <= n; ++ii) {
            temp=wcos*pp[ii-1][j-1]+wsin*pp[ii-1][(j+1)-1];
            pp[ii-1][(j+1)-1]=wcos*pp[ii-1][(j+1)-1]-wsin*pp[ii-1][j-1];
            pp[ii-1][j-1]=temp;
         }
      }
   }
}
}

// -----------------------------------------------------------------------

static void cossin_(double u[], double v[], int j, int low, int high) {

// Combines the vectors u and v, carrying the weight of u[j-1] to v[j-1].
 
double d, temp;
int i;

d= Math.sqrt((u[j-1]*u[j-1])+(v[j-1]*v[j-1]));
wcos=v[j-1]/d;
wsin=u[j-1]/d;
u[j-1]=0.;
v[j-1]=d;
for (i=low; i <= high; ++i) {
   temp = wsin * u[i-1] + wcos * v[i-1];
   u[i-1] = wcos * u[i-1] - wsin * v[i-1];
   v[i-1]=temp;
}
}

// -----------------------------------------------------------------------

static void lowers_(double a[][], int n, int d, double b[]) {
 
// Solves the lower triangular system a*x=b for x.
 
int i, j;

double sum;

for (i=1; i <= n; ++i) {
   sum=0.;
   for (j=1; j < i; ++j) {
      sum += a[d+i-1][j-1]*b[d+j-1];
   }
   b[d+i-1]=(b[d+i-1]-sum)/a[d+i-1][i-1];
}
}

// --------------------------------------------------------------------

static void lowerb_(double a[][], int n, int d, double b[]) {
 
/* Solves the lower triangular system a*x=b for x, square matrix version
   of lowers. */
 
int i, j;

double sum;

for (i=1; i <= n; ++i) {
   sum=0.;
   for (j=1; j < i; ++j) {
      sum += a[d+i-1][j-1] * b[d+j-1];
   }
   b[d+i-1]=(b[d+i-1]-sum) / a[d+i-1][i-1];
}
}

// ---------------------------------------------------------------------

static int covpar_(int m, int k, double c[][], double s[][],
   double cpar[][]) {

/* Computes the covariance matrix of the regression parameters from the
   Paige gls algorithm, wlsqrs_(). */

int i, i1, j;

double check;

double lt[][] = new double[NMWRPAR][NMWRPAR];
double r2[][] = new double[NMWRPAR][NMWRPAR];
double ltinv[][] = new double[NMWRPAR][NMWRPAR];

// Zero the lt and r2 arrays.

for (i = 0; i < k; ++i) {
   for (j = 0; j < k; ++j) {
      lt[i][j] = 0.;
      r2[i][j] = 0.;
   }
}

/* l is a lower triangular matrix stored in the last k rows of c.
   Therefore, extract l**T by transposing on the fly. */

for (i = 0; i < k; ++i) {
   for (j = 0; j < k; ++j) lt[j][i] = c[m - k + i][j];
}

/* r2**T is a k by k lower triangular matrix stored in the lower-right
   corner of s.  Therefore, extract r2 by transposing on the fly. */

for (i = 0; i < k; ++i) {
   for (j = 0; j < k; ++j) r2[j][i] = s[m - k + i][m -k + j];
}

// Compute the inverse of the upper triangular matrix, lt.

trnginv_(k,lt,ltinv);

// Compute r2 * ltinv and store the result in lt.

for (i = 0; i < k; ++i){
   for (j = 0; j < k; ++j) {
      lt[i][j] = 0.;
      for (i1 = 0; i1 < k; ++i1) lt[i][j] += r2[i][i1] * ltinv[i1][j];
   }
}

// cpar = (r2 * ltinv)**T * (r2 * ltinv).

for (i = 0; i < k; ++i){
   for (j = 0; j < k; ++j) {
      cpar[i][j] = 0.;
      for (i1 = 0; i1 < k; ++i1) cpar[i][j] += lt[i1][i] * lt[i1][j];
   }
}
return 0;
}

// -----------------------------------------------------------------

static int trnginv_(int n, double t[][], double s[][]) {

/* Computes the inverse of the upper triangular matrix, t and returns it
   in s.  From Stewart (1973, p. 110). */

int i, j, k;

// Zero the return matrix, s.

for (i = 0; i < n; ++i) {
   for (j = 0; j < n; ++j) s[i][j] = 0.;
}

for (k = n; k >= 1; --k) {
   if (t[k-1][k-1] == 0.) iderr_("0 diagonal element in trnginv");
   s[k-1][k-1] = 1. / t[k-1][k-1];
   for (i = k - 1; i >= 1; --i) {
      s[i-1][k-1] = 0.;
      for (j = i + 1; j <= k; ++j) s[i-1][k-1] += t[i-1][j-1] * s[j-1][k-1];
      if (t[i-1][i-1] == 0.) iderr_("0 diagonal element in trnginv");
      s[i-1][k-1] /= -t[i-1][i-1];
   }
}
return 0;
}
}
