class Tree extends Id {

/* Explanation of classification tree arrays:

   ndeinfo[i-1][j] contains information about node i as follows.
      j=0: number of parent node (the value 0 indicates that this is
	   the root node)
      j=1: -1 if node is a left child, 1 if a right child
      j=2: 0 if node is nonterminal, 1 if terminal, -1 if node has been
	   pruned off
      j=3: node's sample size
      j=4: covariate number to split on
      j=5: left child node number
      j=6: right child node number
      j=7: predicted response category (most frequent category).

   ndevals[i-1][j] for node i:
      j=0: size of subtree i,
      j=1: Deviance of subtree i,
      j=2: Size delta,
      j=3: Deviance delta,
      j=4: Slope (lambda in Chou et al.)
      j=5: Minimum slope (lambda_min),
      j=6: Covariate splitting value.

   ndedesc[i-1][] for node i, the i-1th row is a list of all nodes that
      are descendents of node i.

   ndedat[i-1][] for node i, the i-1th row is a list of all observations
      that have fallen to this node. */

static final int MAXP = 3;
static final int MAXQ = 3;
static final int MAXK = MAXQ + 1;
static final int MAXNMR = 3;
static final int MAXNDES = 20;
static final int MAXN = 100;
static final int MAXLVLS = 10;

static String cvtnme[] = new String[MAXP];
static String catlab[] = new String[MAXK];

// "k" is the number of categories of the response variable.

static int nmprd, nmprdr, nmread, esttyp, nmquanr, nmquan, calc, grpd,
   nmpex, flag, n, nmndes, prune, partit, k;

static int pex[] = new int[MAXP];
static int ndeinfo[][] = new int[MAXNDES][8];
static int ndedesc[][] = new int[MAXNDES][MAXNDES];
static int ndedat[][] = new int[MAXNDES][MAXN];
static int nmlvls[] = new int[MAXP];
static int  qualval[][] = new int[MAXP][MAXLVLS];

static double sigp, rdum, cver, count1, count2, count3, rn;

static double ndevals[][] = new double[MAXNDES][7];
static double spltgrp[][] = new double[MAXNDES][6];
static double prdsr[][] = new double[MAXN][MAXP];
static double prds[][] = new double[MAXN][MAXP];
static double qfreq[][] = new double[MAXP][MAXLVLS];

// ---------------------------------------------------------------------

static int grwtre_(int rspns[], int prune) {

/* Grows a Classification Tree.  Routine assumes exactly 10 levels of a
   qualitative covariate.

   ndedat[i][j] contains the observation number from the full data set
   of the i^th node's j^th observation.  Note that observation numbers
   are repeated many times throughout this array. */

boolean lftgrp = false, dosplit = false;

int i, i1, i2, i3, i4, i5, j, j1, j2, j3, node = 0, minn, nmrnks, level,
   maxfrq, size;

int nprnt[] = new int[MAXQ + 2];
int ndenms[] = new int[MAXN];
int y[][] = new int[MAXN][MAXK];
int rspfrq[] = new int[MAXK];
int rnknm[] = new int[MAXNMR];

double epsln, rn, ndedev, minred, maxred, devred;

double cvrte[] = new double[MAXN];
double ordval[] = new double[MAXN];
double splvls[][] = new double[MAXNMR][6];

// Minimum allowable sample size at a node.

minn = (int) (.04 * ((double) n));

/* Initialize ndedat, the array that defines the tree by assigning
   each observation to a terminal node (as the tree grows, these
   terminal nodes are converted to nonterminal nodes and new terminal
   nodes are defined.  ndevals[][3] < 0 means that node has not been
   visited. */

// All data is at the first node.

for (i = 0; i < n; ++i) ndedat[0][i] = i + 1;
for (i = 0; i < MAXNDES; ++i) {
   ndeinfo[i][2] = -1;
   ndevals[i][3] = -1.;
   for (j = 0; j < MAXNDES; ++j) ndedesc[i][j] = 0;
}
nmndes = 1;
ndeinfo[0][0] = 0;
ndeinfo[0][1] = 0;
ndeinfo[0][2] = 1;
ndeinfo[0][3] = n;
ndevals[0][0] = 1.;

/* Compute Deviance of the 1-node tree and base the minimum reduction
   in Deviance on this value.  Note that minred >= 0, always. */

minred = 1.e-6;

/* Partition terminal nodes until stopping rule is satisfied at all
   terminal nodes.  */

for (;;) {

   /* Split the first node that is terminal and does not satisfy the
      stopping rule.  If there are no such nodes, the tree has been
      grown. */

   for (i = 0; i < nmndes; ++i) {
      dosplit = false;
      if (ndeinfo[i][2] == 1) {
	 if (ndeinfo[i][3] > minn && ndevals[i][3] == -1.) {
	    node = i + 1;
	    dosplit = true;
	 }
      }
   }

   if (!dosplit) {
      break;
   }

   // Compute parent node counts and Deviance calculation constants.

   for (i = 0; i <= k; ++i) nprnt[i] = 0;
   nprnt[0] = ndeinfo[node - 1][3];
   if (nprnt[0] == 0) iderr_("zero nprnt in grwtre");
   rn = (double) nprnt[0];
   ndedev = rn * Math.log(rn);
   for (i = 0; i < nprnt[0]; ++i) {
      ++nprnt[ rspns[ndedat[node - 1][i] - 1] ];
   }
   for (i = 1; i <= k; ++i) {
      rn = (double) nprnt[i];
      if (rn > 1.e-5) ndedev -= rn * Math.log(rn);
   }
   ndevals[node - 1][1] = ndedev;

   /* Find the covariate whose optimal split gives the greatest reduction
      in Deviance. */

   maxred = minred;
   ndevals[node - 1][3] = maxred;
   PREDLOOP: for (j = 1; j <= nmprd; ++j) {

      // Zero out the ranked data array.

      for (i = 0; i < nprnt[0]; ++i) {
	 for (j1 = 0; j1 < k; ++j1) y[i][j1] = 0;
      }

      // Load temporary arrays and compute parent counts.

      for (i = 0; i < nprnt[0]; ++i) {
	 ndenms[i] = ndedat[node - 1][i];
	 cvrte[i] = prds[ndenms[i] - 1][j - 1];
      }
      for (i = 0; i < MAXNMR; ++i) rnknm[i] = 0;

      if (j <= nmquan) {

	 /* Quantitative covariates.  First, define the difference,
	    epsln, necessary to start a new rank. */

         Idsort.idsort_(cvrte, ndenms, 1, nprnt[0]);
         epsln = .001 * (cvrte[nprnt[0] - 1] - cvrte[0]);

         /* Find ranks of the covariate and the response counts within each
	    rank. */

         nmrnks = 1;
         for (i = 0; i < nprnt[0]; ++i) {
            ++rnknm[nmrnks - 1];
            ++y[nmrnks - 1][rspns[ndenms[i] - 1] - 1];
	    ordval[nmrnks - 1] = cvrte[i];
	    if (i < nprnt[0] - 1) {
               if (cvrte[i + 1] - cvrte[i] > epsln) ++nmrnks;
	    }
         }
         if (nmrnks == 1) continue PREDLOOP;

      } else {

	 /* Qualitative covariates.  First, enumerate all possible
	    partitions. */

	 nmrnks = groups_(nprnt[0], cvrte, splvls);
         if (nmrnks == 0) continue PREDLOOP;

	 // Update each group that this observation belongs to.

	 for (i = 0; i < nprnt[0]; ++i) {
	    for (j1 = 0; j1 < nmrnks; ++j1) {
	       for (j2 = 1; j2 <= (int) splvls[j1][0]; ++j2) {
		  if (Math.abs(cvrte[i] - splvls[j1][j2]) < 1.e-6) {
		     ++rnknm[j1];
		     ++y[j1][rspns[ndenms[i] - 1] - 1];
		  }
	       }
	    }
	 }
      }

      /* Check counts within ranks and number of ranks.  Note that
	 as above, for a qualitative covariate, nmrnks is the number
	 of possible 2-group partitions of the covariate's levels. */

      for (i = 0; i < nmrnks; ++i) {
	 if (rnknm[i] < 1) {
	    printf_("covariate= " + j + " n= " + nprnt[0] +
	       " rank= " + (i + 1) + " nmrnks= " + nmrnks);
	    iderr_("zero rank count in grwtre");
	 }
      }

      /* Find optimal partition (split).  The global integeer, "partit"
	 is set in "split." */

      devred = split_(j, nmquan, nmrnks, nprnt, y, ndedev);

      /* If this is the best split so far and has a Deviance reduction
	 bigger than the minimum, save the splitting information. */

      if (devred > maxred) {
	 maxred = devred;

	 // Record the splitting covariate.

	 ndeinfo[node - 1][4] = j;

	 if (j <= nmquan) {

	    // For quantitative covariates.

	    i = 0;
	    do {
	       ndedat[nmndes][i] = ndenms[i];
	       ++i;
	       if (i > nprnt[0]) iderr_("endless do");
	    } while (cvrte[i] <= ordval[partit - 1]);
	    for (j1 = i; j1 < ndeinfo[node - 1][3]; ++j1) {
	       ndedat[ nmndes + 1 ][j1 - i] = ndenms[j1];
	    }

	    // Save the partition sample sizes, and splitting value.

	    ndeinfo[nmndes][3] = i;
	    ndeinfo[nmndes + 1][3] = nprnt[0] - i;
	    ndevals[node - 1][6] = ordval[partit - 1];

	 } else {

	    /* For qualitative covariates.
	       First, record splitting group (partition). */

	    spltgrp[node - 1][0] = splvls[partit - 1][0];
	    for (i = 1; i <= (int) spltgrp[node - 1][0]; ++i) {
	       spltgrp[node - 1][i] = splvls[partit - 1][i];
	    }

	    // Load child node observations.

	    j2 = 0;
	    j3 = 0;
	    for (i = 0; i < nprnt[0]; ++i) {
	       lftgrp = false;
	       for (j1 = 1; j1 <= (int) spltgrp[node - 1][0]; ++j1) {
		  if (Math.abs(cvrte[i] - spltgrp[node - 1][j1])
		      < 1.e-6) lftgrp = true;
	       }
	       if (lftgrp == true) {
	          ndedat[nmndes][j2] = ndenms[i];
	          ++j2;

	       } else {
		  ndedat[nmndes + 1][j3] = ndenms[i];
		  ++j3;
	       }
	    }
            if (j2 + j3 != nprnt[0]) iderr_("j2+j3!=nprnt(0)");

	    // Save the partition sizes.

	    ndeinfo[nmndes][3] = j2;
	    ndeinfo[nmndes + 1][3] = nprnt[0] - j2;
	 }
      }
   } // End of PREDLOOP loop.

   /* If a split was found, set current node to nonterminal, record
      children node numbers, the parent number of these children,
      left-right assignment, terminal status of the children,
      deviance reduction, and increment the number of nodes counter. */

   if (maxred > minred) {
      
      // Update the node counter.

      nmndes += 2;
      if (nmndes > MAXNDES - 2) iderr_("nmndes > MAXNDES - 2 in grwtre");

      // Make node nonterminal, specify child node numbers.

      ndeinfo[node - 1][2] = 0;
      ndeinfo[node - 1][5] = nmndes - 1;
      ndeinfo[node - 1][6] = nmndes;

      /* Specify childrens' parent node (which is node), leftness, rightness
	 of the children, children are terminal nodes. */

      ndeinfo[nmndes - 2][0] = node;
      ndeinfo[nmndes - 1][0] = node;
      ndeinfo[nmndes - 2][1] = -1;
      ndeinfo[nmndes - 1][1] = 1;
      ndeinfo[nmndes - 2][2] = 1;
      ndeinfo[nmndes - 1][2] = 1;

      /* Specify the Deviance reduction due to this split, and the size
	 of the terminal node trees.  */

      ndevals[node - 1][3] = maxred;
      ndevals[nmndes - 2][0] = 1.;
      ndevals[nmndes - 1][0] = 1.;

      // Update all parent node descendent tables and parent node sizes.

      do {
	 size = (int) ndevals[node - 1][0];
	 ndedesc[node - 1][size] = nmndes - 1;
	 ndedesc[node - 1][size + 1] = nmndes;
         ndevals[node - 1][0] += 2.;
         node = ndeinfo[node - 1][0];
      } while (node > 0);
   }
}

// Compute the predicted response at each terminal node.

for (i = 0; i < nmndes; ++i) {
   if (ndeinfo[i][2] == 1) {
      for (j = 0; j < k; ++j) rspfrq[j] = 0;
      for (j = 0; j < ndeinfo[i][3]; ++j) {
	 level = rspns[ndedat[i][j] - 1];
	 if (level < 1 || level > k) iderr_("bad level");
         ++rspfrq[level - 1];
      }
      maxfrq = 0;
      for (j = 0; j < k; ++j) {
	 if (rspfrq[j] > maxfrq) {
	    ndeinfo[i][7] = j + 1;
	    maxfrq = rspfrq[j];
	 }
      }
   }
}

return 0;
}

// ---------------------------------------------------------------------

static int prttre_(int nmndes, int ndeinfo[][], double ndevals[][],
   int ndedesc[][]) {

/* Print the tree and make a SAS NETDRAW input file for plotting the
   tree. */

int i, prntnm;

printf_("nmndes= " + nmndes);
printf_("\nNonterminal Nodes:\n");
printf_(" Node  Left-Child Right-Child   Size   Covariate  Split-Value\n");
for (i = 0; i < MAXNDES; ++i) {
   if (ndeinfo[i][2] == 0) {
      printf_((i + 1) + "  " + ndeinfo[i][5] + "  " + ndeinfo[i][6] +
	 "  " + ndeinfo[i][3] + "   " + ndeinfo[i][4] + "  " +
	 ndevals[i][6]);
   }
}
printf_("\nTerminal Nodes:\n");
printf_(" Node  Size  Prediction\n");
for (i = 0; i < MAXNDES; ++i) {
   if (ndeinfo[i][2] == 1) {
      printf_((i + 1) + "  " + ndeinfo[i][3] + "  " + ndeinfo[i][7]);
   }
}

// Write SAS tree graphics file.

fleopen_(7, "tree.sas", 'w');

fprintf_(7,
   "filename gsasfile 'tree.ps';\n" +
   "goptions device=PSLMONO gaccess=gsasfile reset=global\n" +
   "gunit=cells ftext=swissb htitle=5 htext=5\n" +
   "gsfmode=replace hsize=7.5in vsize=8.in horigin=0.5in vorigin=1.5in\n" +
   "hpos = 150 vpos = 150;\n" +
   "data; input parent $ child $ prntdsc $; cards;");

/* Don't write the root node and repeat the writing of the terminal nodes.
   On the second terminal node write, use the predicted category as the
   label. */

for (i = 1; i < MAXNDES; ++i) {
   if (ndeinfo[i][2] != -1) {
      prntnm = ndeinfo[i][0];
      fprintf_(7, prntnm + (i + 1) + "  " +
	 cvtnme[ ndeinfo[prntnm - 1][4] - 1 ]);
      if (ndeinfo[i][2] == 1) {
         fprintf_(7, (i + 1) + "." + catlab[ ndeinfo[i][7] - 1 ]);
      }
   }
}

fprintf_(7, ";\n" +
   "run;\n" +
   "pattern1 v=e;\n" +
   "proc netdraw graphics;\n" +
   "   actnet / act=parent succ=child id=(prntdsc)\n" +
   "        nodefid nolabel centerid tree compress\n" +
   "        xbetween=8 ybetween=3 arrowhead=3\n" +
   "        rectilinear carcs=black ctext=black;\n" +
   "run;");

fclose_(7, 'w');

return 0;
}

// ---------------------------------------------------------------------

int prntre_(int nmndes, int ndeinfo[][], double ndevals[][],
   int ndedesc[][], int ndedat[][]) {

/* Prunes a classification tree.  Each pass through the outer "while"
   loop below produces a pruned subtree, S that is on the lower
   boundary of the Deviance-size convex hull.  The routine stops
   pruning when a tree that has the desired size,
   specified by the "best" variable (same as in Splus) has been found.
   Because of the proofs in Chou et al., you are assured that no other
   tree of this particular size would give a smaller Deviance. */

boolean done = false;
int i, j, pnode, node, level, maxfrq;

boolean ndeupdt[] = new boolean[MAXNDES];
int rspfrq[] = new int[MAXK];

double infty, nmdel, best;

double trevals[] = new double[4];

/* trevals[j]:   j=0: best size (u_1),
		 j=1: best Deviance (u_2),
		 j=2: delta_1,
		 j=3: delta_2,  */

infty = 1.e20;
best = 57.;
if (best >= (double) (nmndes)) {
   printf_("nmndes= " + nmndes + " best= " + best);
   iderr_("in prntre");
}

// Initialize terminal node values.

for (i = 1; i <= nmndes; ++i) {
   ndeupdt[i - 1] = false;
   if (ndeinfo[i - 1][2] == 1) {
      ndevals[i - 1][2] = 0.;
      ndevals[i - 1][3] = 0.;
      ndevals[i - 1][5] = infty;
      ndeupdt[i - 1] = true;
   }
}

// Initialize interior node values of Size delta, Deviance delta, Lambda.

for (i = 1; i <= nmndes; ++i) {
   if (ndeinfo[i - 1][2] == 0) {

      // Size delta.

      ndevals[i - 1][2] = ndevals[i - 1][0] - 1.;

      /* Use the negative of the maximum Deviance reduction values
	 for the Deviance delta. */

      ndevals[i - 1][3] *= -1.;

      // Lambda.

      ndevals[i - 1][4] = -ndevals[i - 1][3] / ndevals[i - 1][2];
   }
}

// Find lambda_min for each interior node.

do {
   done = true;
   for (i = 1; i <= nmndes; ++i) {
      if (ndeupdt[i - 1] == false) {
         if (ndeinfo[i - 1][2] == 0 &&
             ndeupdt[ ndeinfo[i - 1][5] - 1 ] == true &&
             ndeupdt[ ndeinfo[i - 1][6] - 1 ] == true) {
            ndevals[i - 1][5] = ndevals[i - 1][4];
            if (ndevals[ ndeinfo[i - 1][5] - 1 ][5] < ndevals[i - 1][5]) {
	       ndevals[i - 1][5] = ndevals[ ndeinfo[i - 1][5] - 1 ][5];
            }
            if (ndevals[ ndeinfo[i - 1][6] - 1 ][5] < ndevals[i - 1][5]) {
	       ndevals[i - 1][5] = ndevals[ ndeinfo[i - 1][6] - 1 ][5];
            }
            ndeupdt[i - 1] = true;

         } else {
            done = false;
         }
      }
   }
} while (!done);

// Initialize tree functional vector.

trevals[0] = ndevals[0][0] + ndevals[0][2];
trevals[1] = ndevals[0][1] + ndevals[0][3];

// Pruning loop.

while (ndevals[0][5] < infty) {

   // Starting at the root, find the node with the minimum delta.

   i = 1;
   while (ndevals[i - 1][4] > ndevals[0][5]) {
      if (Math.abs(ndevals[ ndeinfo[i - 1][5] - 1 ][5] - ndevals[0][5]) <
	  1.e-6) {
	 i = ndeinfo[i - 1][5];

      } else {
	 i = ndeinfo[i - 1][6];
      }

      if (ndeinfo[i - 1][2] == 1) {
	 printf_("Terminal node reached while searching for delta_min");
	 iderr_("In prntre_()");
      }
   }
   pnode = i;

   // Store this node's delta values.

   trevals[2] = ndevals[i - 1][2];
   trevals[3] = ndevals[i - 1][3];

   // Prune off this branch, i.e., remove all descendants of node i.

   j = 0;
   node = ndedesc[i - 1][j];
   do {
      ndeinfo[node - 1][2] = -1;
      ++j;
      node = ndedesc[i - 1][j];
   } while (node > 0);

   // Make node i a terminal node and update size values.

   ndevals[i - 1][5] = infty;
   ndeinfo[i - 1][2] = 1;
   nmdel = ndevals[i - 1][0] - 1.;
   nmndes -= (int) nmdel;
   ndevals[i - 1][0] = 1.;

   // Compute predicted response.

   for (j = 0; j < k; ++j) rspfrq[j] = 0;
   for (j = 0; j < ndeinfo[i - 1][3]; ++j) {
      level = rspns[ndedat[i - 1][j] - 1];
      if (level < 1 || level > k) {
	 iderr_("invalid response category in prntre_()");
      }
      ++rspfrq[level - 1];
   }
   maxfrq = 0;
   for (j = 0; j < k; ++j) {
      if (rspfrq[j] > maxfrq) {
         ndeinfo[i - 1][7] = j + 1;
         maxfrq = rspfrq[j];
      }
   }

   // Update all ancestors of this node and retrace back to the root.

   while (i != 1) {

      // Get parent number.

      i = ndeinfo[i - 1][0];

      // Update best deltas.

      ndevals[i - 1][2] -= trevals[2];
      ndevals[i - 1][3] -= trevals[3];

      // Compute lambda.

      ndevals[i - 1][4] = -ndevals[i - 1][3] / ndevals[i - 1][2];

      // Find lambda_min.

      ndevals[i - 1][5] = ndevals[i - 1][4];
      if (ndevals[ ndeinfo[i - 1][5] - 1 ][5] < ndevals[i - 1][5]) {
	 ndevals[i - 1][5] = ndevals[ ndeinfo[i - 1][5] - 1 ][5];
      }
      if (ndevals[ ndeinfo[i - 1][6] - 1 ][5] < ndevals[i - 1][5]) {
	 ndevals[i - 1][5] = ndevals[ ndeinfo[i - 1][6] - 1 ][5];
      }

      // Update size value.

      ndevals[i - 1][0] -= nmdel;
   }

   // Update the tree functionals.

   trevals[0] = trevals[0] - trevals[2];
   trevals[1] = trevals[1] - trevals[3];

   // Return when the optimal tree with the desired size has been found.

   if (ndevals[0][0] < best) return 0;
}
iderr_("Singleton tree reached in prntre_");
return 0;
}

// ---------------------------------------------------------------------

static int prdtre_(int nmquan, int nmndes, double cvrte[], int ndeinfo[][],
   double ndevals[][], double spltgrp[][]) {

/* Predicts the response category at the given covariate location.
   Node 1 is always the root node. */

boolean dochkterm = false;

int i, node = 1;

do {
   
   // Decide left or right.

   if (ndeinfo[node - 1][4] <= nmquan) {
      
      // Quantitative covariate.

      if (cvrte[ndeinfo[node - 1][4] - 1] <= ndevals[node - 1][6]) {
         node = ndeinfo[node - 1][5];

      } else {
         node = ndeinfo[node - 1][6];
      }

   } else {

      // Qualitative covariate.

      for (i = 1; i <= (int) spltgrp[node - 1][0]; ++i) {
	 dochkterm = false;
	 if (Math.abs(cvrte[ndeinfo[node - 1][4] - 1] -
		  spltgrp[node - 1][i]) < 1.e-6) {
	   node = ndeinfo[node - 1][5];
	   dochkterm = true;
	   break;
	 }
      }

      if (!dochkterm) {
         node = ndeinfo[node - 1][6];
      }
   }

   // If this is a terminal node, return predicted value.

   if (ndeinfo[node - 1][2] == 1) {
      return ndeinfo[node - 1][7];
   }
} while (node <= nmndes);

iderr_("prdtre: node= " + node);

return 0;
}

// -----------------------------------------------------------------

static double split_(int cvrte, int nmquan, int nmrnks, int nprnt[],
   int y[][], double ndedev) {

/* Binary partition of y such that the reduction in Deviance is
   maximized.  Returns the deviance reduction value ("devred").

   y holds the response category counts by covariate value rank number
   and is sorted by rank number.  nmrnks is the number of ranks of the
   covariate.  Note that the covariate itself is not needed nor the
   rank boundaries.  This routine assumes there are more than 2 ranks.

   nlft, nrght hold marginal and response category counts:
      0: marginal count,
      1-k: response category count. */

int i, j, j1, j2, nl, nr, nmgrps;

int nlft[] = new int[MAXK + 1];
int nrght[] = new int[MAXK + 1];

double devdff, devred = 0.;

double rnlft[] = new double[MAXK + 1];
double rnrght[] = new double[MAXK + 1];
double rnprnt[] = new double[MAXK + 1];

/* Compute the Deviance reduction from each partition, save partition
   that maximizes this reduction. */

devred = -1.e30;
if (cvrte <= nmquan) nmgrps = nmrnks - 1;
else nmgrps = nmrnks;
for (i = 0; i < nmgrps; ++i) {

   // Compute partition sizes and response category counts.

   for (j = 0; j <= k; ++j) nlft[j] = 0;
   for (j = 1; j <= k; ++j) {
      if (cvrte <= nmquan) {

         // Quantitative covariate.

         for (j1 = 0; j1 <= i; ++j1) nlft[j] += y[j1][j - 1];

      } else {

	 // Qualitative covariate.

	 nlft[j] = y[i][j - 1];
      }
      nrght[j] = nprnt[j] - nlft[j];
      nlft[0] += nlft[j];
   }
   nrght[0] = nprnt[0] - nlft[0];

   // Compute Deviance reduction for this partition.

   for (j = 0; j <= k; ++j) {
      rnlft[j] = (double) nlft[j];
      rnrght[j] = (double) nrght[j];
   }
   devdff = 0.;
   for (j = 1; j <= k; ++j) {
      if (rnlft[j] > 0.) devdff += rnlft[j] * Math.log(rnlft[j]);
      if (rnrght[j] > 0.) devdff += rnrght[j] * Math.log(rnrght[j]);
   }
   if (rnlft[0] > 0.) devdff -= rnlft[0] * Math.log(rnlft[0]);
   if (rnrght[0] > 0.) devdff -= rnrght[0] * Math.log(rnrght[0]);
   devdff = 2. * (ndedev + devdff);

   // Keep track of optimal split.

   if (devdff > devred) {
      partit = i + 1;
      devred = devdff;
      nl = nlft[0];
      nr = nrght[0];
   }
}

return devred;
}

// --------------------------------------------------------------------

static int groups_(int n, double cvrte[], double splvls[][]) {

/* Enumerates the possible binary partitions based on the
   qualitative covariate, "cvrte".  Note that "nmrnks" is the number
   of partitions, NOT the number of ranks. */

int nmwys, nchsk, i, i1, i2, i3, i4, i5, nmvals = 0, nmrnks = 0;
int frqs[] = new int[10];
double lvls[] = new double[10];
double vals[] = new double[10];

// Initialize things.

for (i = 0; i < 10; ++i) {
   frqs[i] = 0;
   vals[i] = (double) (i + 1);
}

// Find the number of unique values in cvrte.

for (i = 0; i < n; ++i) {
   for (i1 = 0; i1 < 10; ++i1) {
      if (Math.abs(cvrte[i] - vals[i1]) < 1.e-6) ++frqs[i1];
   }
}
for (i = 1; i <= 10; ++i) {
   if (frqs[i - 1] > 0) {
      lvls[nmvals] = (double) i;
      ++nmvals;
   }
}
if (nmvals == 1) return nmrnks;

/* Load the array of possible qualitative covariate partitions.
   There is some redundancy in these partitions. */

/* Size 1 groups. */

for (i1 = 1; i1 <= nmvals; ++i1) {
   splvls[nmrnks][0] = 1.;
   splvls[nmrnks][1] = lvls[i1 - 1];
   ++nmrnks;
   if (nmrnks > MAXNMR) iderr_("nmrnks>MAXNMR");
}
if (nmvals <= 3) return nmrnks;

/* Size 2 groups. */

nmwys = 0;
nchsk = Minmax.nchsk_(nmvals, 2);
for (i1 = 1; i1 <= nmvals; ++i1) {
   for (i2 = 1; i2 < i1; ++i2) {
      if (nmwys <= nchsk) {
         splvls[nmrnks][0] = 2.;
         splvls[nmrnks][1] = lvls[i1 - 1];
         splvls[nmrnks][2] = lvls[i2 - 1];
	 ++nmwys;
         ++nmrnks;
         if (nmrnks > MAXNMR) iderr_("nmrnks>MAXNMR");
      }
   }
}
if (nmvals <= 4) return nmrnks;

/* Size 3 groups. */

nmwys = 0;
nchsk = Minmax.nchsk_(nmvals, 3);
for (i1 = 1; i1 <= nmvals; ++i1) {
   for (i2 = 1; i2 < i1; ++i2) {
      for (i3 = 1; i3 < i2; ++i3) {
	 if (nmwys <= nchsk) {
            splvls[nmrnks][0] = 3.;
	    splvls[nmrnks][1] = lvls[i1 - 1];
	    splvls[nmrnks][2] = lvls[i2 - 1];
	    splvls[nmrnks][3] = lvls[i3 - 1];
	    ++nmwys;
	    ++nmrnks;
            if (nmrnks > MAXNMR) iderr_("nmrnks>MAXNMR");
	 }
      }
   }
}
if (nmvals <= 6) return nmrnks;

/* Size 4 groups. */

nmwys = 0;
nchsk = Minmax.nchsk_(nmvals, 4);
for (i1 = 1; i1 <= nmvals; ++i1) {
   for (i2 = 1; i2 < i1; ++i2) {
      for (i3 = 1; i3 < i2; ++i3) {
	 for (i4 = 1; i4 < i3; ++i4) {
	    if (nmwys <= nchsk) {
	       splvls[nmrnks][0] = 4.;
	       splvls[nmrnks][1] = lvls[i1 - 1];
	       splvls[nmrnks][2] = lvls[i2 - 1];
	       splvls[nmrnks][3] = lvls[i3 - 1];
	       splvls[nmrnks][4] = lvls[i4 - 1];
               ++nmwys;
	       ++nmrnks;
               if (nmrnks > MAXNMR) iderr_("nmrnks>MAXNMR");
	    }
	 }
      }
   }
}
if (nmvals <= 8) return nmrnks;

/* Size 5 groups. */

nmwys = 0;
nchsk = Minmax.nchsk_(nmvals, 5);
for (i1 = 1; i1 <= nmvals; ++i1) {
   for (i2 = 1; i2 < i1; ++i2) {
      for (i3 = 1; i3 < i2; ++i3) {
	 for (i4 = 1; i4 < i3; ++i4) {
	    for (i5 = 1; i5 < i4; ++i5) {
	       if (nmwys <= nchsk) {
	          splvls[nmrnks][0] = 5.;
	          splvls[nmrnks][1] = lvls[i1 -1];
	          splvls[nmrnks][2] = lvls[i2 - 1];
	          splvls[nmrnks][3] = lvls[i3 - 1];
	          splvls[nmrnks][4] = lvls[i4 - 1];
	          splvls[nmrnks][5] = lvls[i5 - 1];
		  ++nmwys;
		  ++nmrnks;
                  if (nmrnks > MAXNMR) iderr_("nmrnks>MAXNMR");
	       }
	    }
	 }
      }
   }
}

return nmrnks;
}
}
