/* World's WORST logarithm calculation function:
   Monte Carlo integration of dt/t from 1 to x.

   Author:  Timothy Rolfe

   Transformed to use Java threads for parallel computation:
   each thread computes its component of the random checks.
   The Random generators are instantiated with differing seeds
   --- otherwise we might generate nThreads instances of the
   SAME sequence of x,y pairs.

   Specimen output to System.out (from a quad-processor):

   End of integration range:  2.5
   Total number of shots:  20000000
   Number of threads:  8

   Ln(2.5) = 0.9163958999999999
   Library result:  0.9162907318741551
   Percentage error:  0.01%
   Sequential time required:  28.40

   Ln(2.5) = 0.91625325
   Library result:  0.9162907318741551
   Percentage error:  0.00%
   Parallel time required:  3.98
   Speed-up:  7.13
*/
import java.util.Scanner;
import java.util.Random;
import java.text.DecimalFormat;

public class LnCalc
{
   static Scanner console = new Scanner(System.in);

   static double f(double x)
   {  return 1/x;  }

   static public long scatter (long shots, double front, double range,
                 double top, Random generator)
   {
      long hits = 0;

      while ( shots-- > 0 )
      {
         double xpos = generator.nextDouble() * range + front,
                ypos = generator.nextDouble() * top;

         if ( f(xpos) >= ypos )
            hits++;
      }
      return hits;
   }

   public static void main ( String[] args ) throws Exception
   {
      long   shots;               // Total number of "pellets" in shotgun
      long   remaining;
      long   start, finish;       // Capture milliseconds of calculation
      double seqTime, parTime;    // Time for sequential and parallel
      double front = 1.0,         // Integration always starts at 1.0
             back,                // End of integration region
             range,               // Scatter actually wants range
             top;                 // Largest allowed value for f(x)
      Random generator = new Random(); // Passed to scatter()
      double result,              // Calculated value
             libValue;            // Result from Math.log()
      String txtIn;               // For user dialog
      int    nThreads, thread;
      LnThread[] engine;          // Compute engines for parallel comp.

      System.out.print ("End of integration range:  ");
      if ( args.length < 1 )
         back = console.nextDouble();
      else
      {
         txtIn = args[0];
         back = Double.parseDouble(txtIn.trim());
         System.out.println (back);
      }

      System.out.print ("Total number of shots:  ");
      if ( args.length < 2 )
         shots = console.nextLong();
      else
      {
         txtIn = args[1];
         shots = Long.parseLong(txtIn.trim());
         System.out.println (shots);
      }

      System.out.print ("Number of threads:  ");
      if ( args.length < 3 )
      {  nThreads = console.nextInt();  console.nextLine();  }
      else
      {
         txtIn = args[2];
         nThreads = Integer.parseInt(txtIn.trim());
         System.out.println (nThreads);
      }

      top = ( back >= front ) ? f(front) : f(back);

   // Actual computation:  scatter() returns number of shots below f(x)
      range  = back-front;
      start  = System.currentTimeMillis();
      result = (double) scatter (shots, front, range, top, generator);
      result = result / shots * range * top;
      finish = System.currentTimeMillis();
      libValue = Math.log(back);
      seqTime  = (finish-start)*1.0E-03;

      System.out.printf("\nLn(%2.2f) = %8.8g\n", back, result);
      System.out.printf("Library result:  %.8g\n", libValue);
      System.out.printf("Percentage error:  %.8g\n",
                         Math.abs(result - libValue) * 100 / libValue);
      System.out.printf("sequential time required:  %.4f\n", seqTime);
      System.err.println("Beginning parallel execution.");

      start  = System.currentTimeMillis();
      engine = new LnThread[nThreads];

      remaining = shots;
      for ( thread = 0; thread < nThreads; thread++ )
      {
         long this_set = remaining / (nThreads - thread);
      // Attempting seeds that will generate DIFFERENT sequences.
         long seed = (System.currentTimeMillis() % 1000 +
                      thread*thread*thread) * (thread+1);

         if ( remaining % (nThreads - thread) != 0 )
            this_set++;

         remaining -= this_set;

         engine[thread] = new LnThread
            (this_set, front, back, top, seed, thread);
/**/  // Remove * and / to join loops
      }

      for ( thread = 0; thread < nThreads; thread++ )
      {
/**/
         engine[thread].start();
      }

      for ( result = thread = 0; thread < nThreads; thread++ )
      {
         engine[thread].join();
         result += engine[thread].hits;
      }

      result = result / shots * range * top;
      finish = System.currentTimeMillis();
      parTime  = (finish-start)*1.0E-03;

      System.out.printf("\nLn(%.2f) = %.8g\n", back, result);
      System.out.printf("Library result:  %.8g\n", libValue);
      System.out.printf("Percentage error:  %.8g\n",
                         Math.abs(result - libValue) * 100 / libValue);
      System.out.printf("parllel time required:  %.4f\n", parTime);
      System.out.printf("Speed-up:  %.2f\n", seqTime/parTime);

      System.out.print("Press <Enter> to exit:  ");
      console.nextLine();
  }

    static class LnThread extends Thread
    {
       static boolean DEBUG = false;  // ? ? Show subproblem specs ? ?

       public long hits = 0;
       long   shots;
       double front, range, top;
       Random generator;
       int    myPlace;

       LnThread ( long shots, double front, double back,
                  double top, long seed, int myPlace )
       {
          this.shots = shots;
          this.front = front;
          this.range = back - front;
          this.top   = top;
          this.myPlace = myPlace;
          generator = new Random(seed);
          if ( DEBUG )
             System.out.printf(
                "Instance %d is doing %d shots of the total\nGenerator check:  %#x\n",
                myPlace, shots, generator.nextInt());
       }

       public void run()
       {
          hits = LnCalc.scatter(shots, front, range, top, generator);
       }
    }
}
